(function () {

////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                        //
// packages/observe-sequence/observe_sequence_tests.js                                                    //
//                                                                                                        //
////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                          //
// Run a function named `run` which modifies a sequence. While it                                         // 1
// executes, observe changes to the sequence and accumulate them in an                                    // 2
// array, canonicalizing as necessary. Then make sure the results are                                     // 3
// the same as passed in `expectedCallbacks`. In items in                                                 // 4
// `expectedCallbacks`, allow for special values of the form {NOT:                                        // 5
// "foo"}, which match anything other than "foo".                                                         // 6
//                                                                                                        // 7
// @param test {Object} as passed to Tinytest.add                                                         // 8
// @param sequenceFunc {Function(): sequence type}                                                        // 9
// @param run {Function()} modify the sequence or cause sequenceFunc                                      // 10
//     to be recomupted                                                                                   // 11
// @param expectedCallbacks {Array}                                                                       // 12
//     elements are objects eg {addedAt: [array of arguments]}                                            // 13
// @param numExpectedWarnings {Number}                                                                    // 14
runOneObserveSequenceTestCase = function (test, sequenceFunc,                                             // 15
                                          run, expectedCallbacks,                                         // 16
                                          numExpectedWarnings) {                                          // 17
  if (numExpectedWarnings)                                                                                // 18
    ObserveSequence._suppressWarnings += numExpectedWarnings;                                             // 19
                                                                                                          // 20
  var firedCallbacks = [];                                                                                // 21
  var handle = ObserveSequence.observe(sequenceFunc, {                                                    // 22
    addedAt: function () {                                                                                // 23
      firedCallbacks.push({addedAt: _.toArray(arguments)});                                               // 24
    },                                                                                                    // 25
    changedAt: function () {                                                                              // 26
      var obj = {changedAt: _.toArray(arguments)};                                                        // 27
                                                                                                          // 28
      // Browsers are inconsistent about the order in which 'changedAt'                                   // 29
      // callbacks fire. To ensure consistent behavior of these tests,                                    // 30
      // we can't simply push `obj` at the end of `firedCallbacks` as                                     // 31
      // we do for the other callbacks. Instead, we use insertion sort                                    // 32
      // to place `obj` in a canonical position within the chunk of                                       // 33
      // contiguously recently fired 'changedAt' callbacks.                                               // 34
      for (var i = firedCallbacks.length; i > 0; i--) {                                                   // 35
                                                                                                          // 36
        var compareTo = firedCallbacks[i - 1];                                                            // 37
        if (!compareTo.changedAt)                                                                         // 38
          break;                                                                                          // 39
                                                                                                          // 40
        if (EJSON.stringify(compareTo, {canonical: true}) <                                               // 41
            EJSON.stringify(obj, {canonical: true}))                                                      // 42
          break;                                                                                          // 43
      }                                                                                                   // 44
                                                                                                          // 45
      firedCallbacks.splice(i, 0, obj);                                                                   // 46
    },                                                                                                    // 47
    removedAt: function () {                                                                              // 48
      firedCallbacks.push({removedAt: _.toArray(arguments)});                                             // 49
    },                                                                                                    // 50
    movedTo: function () {                                                                                // 51
      firedCallbacks.push({movedTo: _.toArray(arguments)});                                               // 52
    }                                                                                                     // 53
  });                                                                                                     // 54
                                                                                                          // 55
  run();                                                                                                  // 56
  Deps.flush();                                                                                           // 57
  handle.stop();                                                                                          // 58
                                                                                                          // 59
  test.equal(ObserveSequence._suppressWarnings, 0);                                                       // 60
  test.equal(ObserveSequence._loggedWarnings, 0);                                                         // 61
  ObserveSequence._loggedWarnings = 0;                                                                    // 62
                                                                                                          // 63
  // any expected argument this is `{NOT: "foo"}`, should match any                                       // 64
  // corresponding value in the fired callbacks other than "foo". so,                                     // 65
  // assert non-equality and then replace the appropriate entries in                                      // 66
  // the 'firedCallbacks' array with `{NOT: "foo"}` before calling                                        // 67
  // `test.equal` below.                                                                                  // 68
  var commonLength = Math.min(firedCallbacks.length, expectedCallbacks.length);                           // 69
  for (var i = 0; i < commonLength; i++) {                                                                // 70
    var callback = expectedCallbacks[i];                                                                  // 71
    if (_.keys(callback).length !== 1)                                                                    // 72
      throw new Error("Callbacks should be objects with one key, eg `addedAt`");                          // 73
    var callbackName = _.keys(callback)[0];                                                               // 74
    var args = _.values(callback)[0];                                                                     // 75
    _.each(args, function (arg, argIndex) {                                                               // 76
      if (arg && typeof arg === 'object' &&                                                               // 77
          'NOT' in arg &&                                                                                 // 78
          firedCallbacks[i][callbackName]) {                                                              // 79
        test.notEqual(firedCallbacks[i][callbackName][argIndex],                                          // 80
                      arg.NOT, "Should be NOT " + arg.NOT);                                               // 81
        firedCallbacks[i][callbackName][argIndex] = arg;                                                  // 82
      }                                                                                                   // 83
    });                                                                                                   // 84
  }                                                                                                       // 85
                                                                                                          // 86
  var compress = function (str) {                                                                         // 87
    return str.replace(/\[\n\s*/gm, "[").replace(/\{\n\s*/gm, "{").                                       // 88
      replace(/\n\s*\]/gm, "]").replace(/\n\s*\}/gm, "}");                                                // 89
  };                                                                                                      // 90
                                                                                                          // 91
  test.equal(compress(EJSON.stringify(firedCallbacks, {canonical: true, indent: true})),                  // 92
             compress(EJSON.stringify(expectedCallbacks, {canonical: true, indent: true})));              // 93
};                                                                                                        // 94
                                                                                                          // 95
Tinytest.add('observe-sequence - initial data for all sequence types', function (test) {                  // 96
  runOneObserveSequenceTestCase(test, function () {                                                       // 97
    return null;                                                                                          // 98
  }, function () {}, []);                                                                                 // 99
                                                                                                          // 100
  runOneObserveSequenceTestCase(test, function () {                                                       // 101
    return [];                                                                                            // 102
  }, function () {}, []);                                                                                 // 103
                                                                                                          // 104
  runOneObserveSequenceTestCase(test, function () {                                                       // 105
    return [{foo: 1}, {bar: 2}];                                                                          // 106
  }, function () {}, [                                                                                    // 107
    {addedAt: [0, {foo: 1}, 0, null]},                                                                    // 108
    {addedAt: [1, {bar: 2}, 1, null]}                                                                     // 109
  ]);                                                                                                     // 110
                                                                                                          // 111
  runOneObserveSequenceTestCase(test, function () {                                                       // 112
    return [{_id: "13", foo: 1}, {_id: "37", bar: 2}];                                                    // 113
  }, function () {}, [                                                                                    // 114
    {addedAt: ["13", {_id: "13", foo: 1}, 0, null]},                                                      // 115
    {addedAt: ["37", {_id: "37", bar: 2}, 1, null]}                                                       // 116
  ]);                                                                                                     // 117
                                                                                                          // 118
  runOneObserveSequenceTestCase(test, function () {                                                       // 119
    var coll = new Meteor.Collection(null);                                                               // 120
    coll.insert({_id: "13", foo: 1});                                                                     // 121
    coll.insert({_id: "37", bar: 2});                                                                     // 122
    var cursor = coll.find({}, {sort: {_id: 1}});                                                         // 123
    return cursor;                                                                                        // 124
  }, function () {}, [                                                                                    // 125
    {addedAt: ["13", {_id: "13", foo: 1}, 0, null]},                                                      // 126
    {addedAt: ["37", {_id: "37", bar: 2}, 1, null]}                                                       // 127
  ]);                                                                                                     // 128
                                                                                                          // 129
  // shouldn't break on array with duplicate _id's, and the ids sent                                      // 130
  // in the callbacks should be distinct                                                                  // 131
  runOneObserveSequenceTestCase(test, function () {                                                       // 132
    return [                                                                                              // 133
      {_id: "13", foo: 1},                                                                                // 134
      {_id: "13", foo: 2}                                                                                 // 135
    ];                                                                                                    // 136
  }, function () {}, [                                                                                    // 137
    {addedAt: ["13", {_id: "13", foo: 1}, 0, null]},                                                      // 138
    {addedAt: [{NOT: "13"}, {_id: "13", foo: 2}, 1, null]}                                                // 139
  ], /*numExpectedWarnings = */1);                                                                        // 140
});                                                                                                       // 141
                                                                                                          // 142
Tinytest.add('observe-sequence - array to other array', function (test) {                                 // 143
  var dep = new Deps.Dependency;                                                                          // 144
  var seq = [{_id: "13", foo: 1}, {_id: "37", bar: 2}];                                                   // 145
                                                                                                          // 146
  runOneObserveSequenceTestCase(test, function () {                                                       // 147
    dep.depend();                                                                                         // 148
    return seq;                                                                                           // 149
  }, function () {                                                                                        // 150
    seq = [{_id: "13", foo: 1}, {_id: "38", bar: 2}];                                                     // 151
    dep.changed();                                                                                        // 152
  }, [                                                                                                    // 153
    {addedAt: ["13", {_id: "13", foo: 1}, 0, null]},                                                      // 154
    {addedAt: ["37", {_id: "37", bar: 2}, 1, null]},                                                      // 155
    {removedAt: ["37", {_id: "37", bar: 2}, 1]},                                                          // 156
    {addedAt: ["38", {_id: "38", bar: 2}, 1, null]},                                                      // 157
    {changedAt: ["13", {_id: "13", foo: 1}, {_id: "13", foo: 1}, 0]}                                      // 158
  ]);                                                                                                     // 159
});                                                                                                       // 160
                                                                                                          // 161
Tinytest.add('observe-sequence - array to other array, strings', function (test) {                        // 162
  var dep = new Deps.Dependency;                                                                          // 163
  var seq = ["A", "B"];                                                                                   // 164
                                                                                                          // 165
  runOneObserveSequenceTestCase(test, function () {                                                       // 166
    dep.depend();                                                                                         // 167
    return seq;                                                                                           // 168
  }, function () {                                                                                        // 169
    seq = ["B", "C"];                                                                                     // 170
    dep.changed();                                                                                        // 171
  }, [                                                                                                    // 172
    {addedAt: ["-A", "A", 0, null]},                                                                      // 173
    {addedAt: ["-B", "B", 1, null]},                                                                      // 174
    {removedAt: ["-A", "A", 0]},                                                                          // 175
    {addedAt: ["-C", "C", 1, null]}                                                                       // 176
  ]);                                                                                                     // 177
});                                                                                                       // 178
                                                                                                          // 179
Tinytest.add('observe-sequence - array to other array, objects without ids', function (test) {            // 180
  var dep = new Deps.Dependency;                                                                          // 181
  var seq = [{foo: 1}, {bar: 2}];                                                                         // 182
                                                                                                          // 183
  runOneObserveSequenceTestCase(test, function () {                                                       // 184
    dep.depend();                                                                                         // 185
    return seq;                                                                                           // 186
  }, function () {                                                                                        // 187
    seq = [{foo: 2}];                                                                                     // 188
    dep.changed();                                                                                        // 189
  }, [                                                                                                    // 190
    {addedAt: [0, {foo: 1}, 0, null]},                                                                    // 191
    {addedAt: [1, {bar: 2}, 1, null]},                                                                    // 192
    {removedAt: [1, {bar: 2}, 1]},                                                                        // 193
    {changedAt: [0, {foo: 2}, {foo: 1}, 0]}                                                               // 194
  ]);                                                                                                     // 195
});                                                                                                       // 196
                                                                                                          // 197
Tinytest.add('observe-sequence - array to other array, changes', function (test) {                        // 198
  var dep = new Deps.Dependency;                                                                          // 199
  var seq = [{_id: "13", foo: 1}, {_id: "37", bar: 2}, {_id: "42", baz: 42}];                             // 200
                                                                                                          // 201
  runOneObserveSequenceTestCase(test, function () {                                                       // 202
    dep.depend();                                                                                         // 203
    return seq;                                                                                           // 204
  }, function () {                                                                                        // 205
    seq = [{_id: "13", foo: 1}, {_id: "38", bar: 2}, {_id: "42", baz: 43}];                               // 206
    dep.changed();                                                                                        // 207
  }, [                                                                                                    // 208
    {addedAt: ["13", {_id: "13", foo: 1}, 0, null]},                                                      // 209
    {addedAt: ["37", {_id: "37", bar: 2}, 1, null]},                                                      // 210
    {addedAt: ["42", {_id: "42", baz: 42}, 2, null]},                                                     // 211
    {removedAt: ["37", {_id: "37", bar: 2}, 1]},                                                          // 212
    {addedAt: ["38", {_id: "38", bar: 2}, 1, "42"]},                                                      // 213
    // change fires for all elements, because we don't diff the actual                                    // 214
    // objects.                                                                                           // 215
    {changedAt: ["13", {_id: "13", foo: 1}, {_id: "13", foo: 1}, 0]},                                     // 216
    {changedAt: ["42", {_id: "42", baz: 43}, {_id: "42", baz: 42}, 2]}                                    // 217
  ]);                                                                                                     // 218
});                                                                                                       // 219
                                                                                                          // 220
Tinytest.add('observe-sequence - array to other array, movedTo', function (test) {                        // 221
  var dep = new Deps.Dependency;                                                                          // 222
  var seq = [{_id: "13", foo: 1}, {_id: "37", bar: 2}, {_id: "42", baz: 42}, {_id: "43", baz: 43}];       // 223
                                                                                                          // 224
  runOneObserveSequenceTestCase(test, function () {                                                       // 225
    dep.depend();                                                                                         // 226
    return seq;                                                                                           // 227
  }, function () {                                                                                        // 228
    seq = [{_id: "43", baz: 43}, {_id: "37", bar: 2}, {_id: "42", baz: 42}, {_id: "13", foo: 1}];         // 229
    dep.changed();                                                                                        // 230
  }, [                                                                                                    // 231
    {addedAt: ["13", {_id: "13", foo: 1}, 0, null]},                                                      // 232
    {addedAt: ["37", {_id: "37", bar: 2}, 1, null]},                                                      // 233
    {addedAt: ["42", {_id: "42", baz: 42}, 2, null]},                                                     // 234
    {addedAt: ["43", {_id: "43", baz: 43}, 3, null]},                                                     // 235
                                                                                                          // 236
    {movedTo: ["43", {_id: "43", baz: 43}, 3, 1, "37"]},                                                  // 237
    {movedTo: ["13", {_id: "13", foo: 1}, 0, 3, null]},                                                   // 238
                                                                                                          // 239
    {changedAt: ["13", {_id: "13", foo: 1}, {_id: "13", foo: 1}, 3]},                                     // 240
    {changedAt: ["37", {_id: "37", bar: 2}, {_id: "37", bar: 2}, 1]},                                     // 241
    {changedAt: ["42", {_id: "42", baz: 42}, {_id: "42", baz: 42}, 2]},                                   // 242
    {changedAt: ["43", {_id: "43", baz: 43}, {_id: "43", baz: 43}, 0]}                                    // 243
  ]);                                                                                                     // 244
});                                                                                                       // 245
                                                                                                          // 246
Tinytest.add('observe-sequence - array to null', function (test) {                                        // 247
  var dep = new Deps.Dependency;                                                                          // 248
  var seq = [{_id: "13", foo: 1}, {_id: "37", bar: 2}];                                                   // 249
                                                                                                          // 250
  runOneObserveSequenceTestCase(test, function () {                                                       // 251
    dep.depend();                                                                                         // 252
    return seq;                                                                                           // 253
  }, function () {                                                                                        // 254
    seq = null;                                                                                           // 255
    dep.changed();                                                                                        // 256
  }, [                                                                                                    // 257
    {addedAt: ["13", {_id: "13", foo: 1}, 0, null]},                                                      // 258
    {addedAt: ["37", {_id: "37", bar: 2}, 1, null]},                                                      // 259
    {removedAt: ["13", {_id: "13", foo: 1}, 0]},                                                          // 260
    {removedAt: ["37", {_id: "37", bar: 2}, 0]}                                                           // 261
  ]);                                                                                                     // 262
});                                                                                                       // 263
                                                                                                          // 264
Tinytest.add('observe-sequence - array to cursor', function (test) {                                      // 265
  var dep = new Deps.Dependency;                                                                          // 266
  var seq = [{_id: "13", foo: 1}, {_id: "37", bar: 2}];                                                   // 267
                                                                                                          // 268
  runOneObserveSequenceTestCase(test, function () {                                                       // 269
    dep.depend();                                                                                         // 270
    return seq;                                                                                           // 271
  }, function () {                                                                                        // 272
    var coll = new Meteor.Collection(null);                                                               // 273
    coll.insert({_id: "13", foo: 1});                                                                     // 274
    coll.insert({_id: "38", bar: 2});                                                                     // 275
    var cursor = coll.find({}, {sort: {_id: 1}});                                                         // 276
    seq = cursor;                                                                                         // 277
    dep.changed();                                                                                        // 278
  }, [                                                                                                    // 279
    {addedAt: ["13", {_id: "13", foo: 1}, 0, null]},                                                      // 280
    {addedAt: ["37", {_id: "37", bar: 2}, 1, null]},                                                      // 281
    {removedAt: ["37", {_id: "37", bar: 2}, 1]},                                                          // 282
    {addedAt: ["38", {_id: "38", bar: 2}, 1, null]},                                                      // 283
    {changedAt: ["13", {_id: "13", foo: 1}, {_id: "13", foo: 1}, 0]}                                      // 284
  ]);                                                                                                     // 285
});                                                                                                       // 286
                                                                                                          // 287
                                                                                                          // 288
Tinytest.add('observe-sequence - cursor to null', function (test) {                                       // 289
  var dep = new Deps.Dependency;                                                                          // 290
  var coll = new Meteor.Collection(null);                                                                 // 291
  coll.insert({_id: "13", foo: 1});                                                                       // 292
  coll.insert({_id: "37", bar: 2});                                                                       // 293
  var cursor = coll.find({}, {sort: {_id: 1}});                                                           // 294
  var seq = cursor;                                                                                       // 295
                                                                                                          // 296
  runOneObserveSequenceTestCase(test, function () {                                                       // 297
    dep.depend();                                                                                         // 298
    return seq;                                                                                           // 299
  }, function () {                                                                                        // 300
    seq = null;                                                                                           // 301
    dep.changed();                                                                                        // 302
  }, [                                                                                                    // 303
    {addedAt: ["13", {_id: "13", foo: 1}, 0, null]},                                                      // 304
    {addedAt: ["37", {_id: "37", bar: 2}, 1, null]},                                                      // 305
    {removedAt: ["13", {_id: "13", foo: 1}, 0]},                                                          // 306
    {removedAt: ["37", {_id: "37", bar: 2}, 0]}                                                           // 307
  ]);                                                                                                     // 308
});                                                                                                       // 309
                                                                                                          // 310
Tinytest.add('observe-sequence - cursor to array', function (test) {                                      // 311
  var dep = new Deps.Dependency;                                                                          // 312
  var coll = new Meteor.Collection(null);                                                                 // 313
  coll.insert({_id: "13", foo: 1});                                                                       // 314
  var cursor = coll.find({}, {sort: {_id: 1}});                                                           // 315
  var seq = cursor;                                                                                       // 316
                                                                                                          // 317
  runOneObserveSequenceTestCase(test, function () {                                                       // 318
    dep.depend();                                                                                         // 319
    return seq;                                                                                           // 320
  }, function () {                                                                                        // 321
    coll.insert({_id: "37", bar: 2});                                                                     // 322
    seq = [{_id: "13", foo: 1}, {_id: "38", bar: 2}];                                                     // 323
    dep.changed();                                                                                        // 324
  }, [                                                                                                    // 325
    {addedAt: ["13", {_id: "13", foo: 1}, 0, null]},                                                      // 326
    {addedAt: ["37", {_id: "37", bar: 2}, 1, null]},                                                      // 327
    {removedAt: ["37", {_id: "37", bar: 2}, 1]},                                                          // 328
    {addedAt: ["38", {_id: "38", bar: 2}, 1, null]},                                                      // 329
    {changedAt: ["13", {_id: "13", foo: 1}, {_id: "13", foo: 1}, 0]}                                      // 330
  ]);                                                                                                     // 331
});                                                                                                       // 332
                                                                                                          // 333
Tinytest.add('observe-sequence - cursor', function (test) {                                               // 334
  var coll = new Meteor.Collection(null);                                                                 // 335
  coll.insert({_id: "13", rank: 1});                                                                      // 336
  var cursor = coll.find({}, {sort: {rank: 1}});                                                          // 337
  var seq = cursor;                                                                                       // 338
                                                                                                          // 339
  runOneObserveSequenceTestCase(test, function () {                                                       // 340
    return seq;                                                                                           // 341
  }, function () {                                                                                        // 342
    coll.insert({_id: "37", rank: 2});                                                                    // 343
    coll.insert({_id: "77", rank: 3});                                                                    // 344
    coll.remove({_id: "37"});                           // should fire a 'removedAt' callback             // 345
    coll.insert({_id: "11", rank: 0});                  // should fire an 'addedAt' callback              // 346
    coll.update({_id: "13"}, {$set: {updated: true}});  // should fire an 'changedAt' callback            // 347
    coll.update({_id: "77"}, {$set: {rank: -1}});       // should fire 'changedAt' and 'movedTo' callback // 348
  }, [                                                                                                    // 349
    // this case must not fire spurious calls as the array to array                                       // 350
    // case does. otherwise, the entire power of cursors is lost in                                       // 351
    // blaze.                                                                                             // 352
    {addedAt: ["13", {_id: "13", rank: 1}, 0, null]},                                                     // 353
    {addedAt: ["37", {_id: "37", rank: 2}, 1, null]},                                                     // 354
    {addedAt: ["77", {_id: "77", rank: 3}, 2, null]},                                                     // 355
    {removedAt: ["37", {_id: "37", rank: 2}, 1]},                                                         // 356
    {addedAt: ["11", {_id: "11", rank: 0}, 0, "13"]},                                                     // 357
    {changedAt: ["13", {_id: "13", rank: 1, updated: true}, {_id: "13", rank: 1}, 1]},                    // 358
    {changedAt: ["77", {_id: "77", rank: -1}, {_id: "77", rank: 3}, 2]},                                  // 359
    {movedTo: ["77", {_id: "77", rank: -1}, 2, 0, "11"]}                                                  // 360
  ]);                                                                                                     // 361
});                                                                                                       // 362
                                                                                                          // 363
Tinytest.add('observe-sequence - cursor to other cursor', function (test) {                               // 364
  var dep = new Deps.Dependency;                                                                          // 365
  var coll = new Meteor.Collection(null);                                                                 // 366
  coll.insert({_id: "13", foo: 1});                                                                       // 367
  var cursor = coll.find({}, {sort: {_id: 1}});                                                           // 368
  var seq = cursor;                                                                                       // 369
                                                                                                          // 370
  runOneObserveSequenceTestCase(test, function () {                                                       // 371
    dep.depend();                                                                                         // 372
    return seq;                                                                                           // 373
  }, function () {                                                                                        // 374
    coll.insert({_id: "37", bar: 2});                                                                     // 375
                                                                                                          // 376
    var newColl = new Meteor.Collection(null);                                                            // 377
    newColl.insert({_id: "13", foo: 1});                                                                  // 378
    newColl.insert({_id: "38", bar: 2});                                                                  // 379
    var newCursor = newColl.find({}, {sort: {_id: 1}});                                                   // 380
    seq = newCursor;                                                                                      // 381
    dep.changed();                                                                                        // 382
  }, [                                                                                                    // 383
    {addedAt: ["13", {_id: "13", foo: 1}, 0, null]},                                                      // 384
    {addedAt: ["37", {_id: "37", bar: 2}, 1, null]},                                                      // 385
    {removedAt: ["37", {_id: "37", bar: 2}, 1]},                                                          // 386
    {addedAt: ["38", {_id: "38", bar: 2}, 1, null]},                                                      // 387
    {changedAt: ["13", {_id: "13", foo: 1}, {_id: "13", foo: 1}, 0]}                                      // 388
  ]);                                                                                                     // 389
});                                                                                                       // 390
                                                                                                          // 391
Tinytest.add('observe-sequence - cursor to other cursor with transform', function (test) {                // 392
  var dep = new Deps.Dependency;                                                                          // 393
  var transform = function(doc) {                                                                         // 394
    return _.extend({idCopy: doc._id}, doc);                                                              // 395
  };                                                                                                      // 396
                                                                                                          // 397
  var coll = new Meteor.Collection(null, {transform: transform});                                         // 398
  coll.insert({_id: "13", foo: 1});                                                                       // 399
  var cursor = coll.find({}, {sort: {_id: 1}});                                                           // 400
  var seq = cursor;                                                                                       // 401
                                                                                                          // 402
  runOneObserveSequenceTestCase(test, function () {                                                       // 403
    dep.depend();                                                                                         // 404
    return seq;                                                                                           // 405
  }, function () {                                                                                        // 406
    coll.insert({_id: "37", bar: 2});                                                                     // 407
                                                                                                          // 408
    var newColl = new Meteor.Collection(null, {transform: transform});                                    // 409
    newColl.insert({_id: "13", foo: 1});                                                                  // 410
    newColl.insert({_id: "38", bar: 2});                                                                  // 411
    var newCursor = newColl.find({}, {sort: {_id: 1}});                                                   // 412
    seq = newCursor;                                                                                      // 413
    dep.changed();                                                                                        // 414
  }, [                                                                                                    // 415
    {addedAt: ["13", {_id: "13", foo: 1, idCopy: "13"}, 0, null]},                                        // 416
    {addedAt: ["37", {_id: "37", bar: 2, idCopy: "37"}, 1, null]},                                        // 417
    {removedAt: ["37", {_id: "37", bar: 2, idCopy: "37"}, 1]},                                            // 418
    {addedAt: ["38", {_id: "38", bar: 2, idCopy: "38"}, 1, null]},                                        // 419
    {changedAt: ["13", {_id: "13", foo: 1, idCopy: "13"}, {_id: "13", foo: 1, idCopy: "13"}, 0]}          // 420
  ]);                                                                                                     // 421
});                                                                                                       // 422
                                                                                                          // 423
Tinytest.add('observe-sequence - cursor to same cursor', function (test) {                                // 424
  var coll = new Meteor.Collection(null);                                                                 // 425
  coll.insert({_id: "13", rank: 1});                                                                      // 426
  var cursor = coll.find({}, {sort: {rank: 1}});                                                          // 427
  var seq = cursor;                                                                                       // 428
  var dep = new Deps.Dependency;                                                                          // 429
                                                                                                          // 430
  runOneObserveSequenceTestCase(test, function () {                                                       // 431
    dep.depend();                                                                                         // 432
    return seq;                                                                                           // 433
  }, function () {                                                                                        // 434
    coll.insert({_id: "24", rank: 2});                                                                    // 435
    dep.changed();                                                                                        // 436
    Deps.flush();                                                                                         // 437
    coll.insert({_id: "78", rank: 3});                                                                    // 438
  }, [                                                                                                    // 439
    {addedAt: ["13", {_id: "13", rank: 1}, 0, null]},                                                     // 440
    {addedAt: ["24", {_id: "24", rank: 2}, 1, null]},                                                     // 441
    // even if the cursor changes to the same cursor, we diff to see if we                                // 442
    // missed anything during the invalidation, which leads to these                                      // 443
    // 'changedAt' events.                                                                                // 444
    {changedAt: ["13", {_id: "13", rank: 1}, {_id: "13", rank: 1}, 0]},                                   // 445
    {changedAt: ["24", {_id: "24", rank: 2}, {_id: "24", rank: 2}, 1]},                                   // 446
    {addedAt: ["78", {_id: "78", rank: 3}, 2, null]}                                                      // 447
  ]);                                                                                                     // 448
});                                                                                                       // 449
                                                                                                          // 450
Tinytest.add('observe-sequence - string arrays', function (test) {                                        // 451
  var seq = ['A', 'B'];                                                                                   // 452
  var dep = new Deps.Dependency;                                                                          // 453
                                                                                                          // 454
  runOneObserveSequenceTestCase(test, function () {                                                       // 455
    dep.depend();                                                                                         // 456
    return seq;                                                                                           // 457
  }, function () {                                                                                        // 458
    seq = ['B', 'C'];                                                                                     // 459
    dep.changed();                                                                                        // 460
  }, [                                                                                                    // 461
    {addedAt: ['-A', 'A', 0, null]},                                                                      // 462
    {addedAt: ['-B', 'B', 1, null]},                                                                      // 463
    {removedAt: ['-A', 'A', 0]},                                                                          // 464
    {addedAt: ['-C', 'C', 1, null]}                                                                       // 465
  ]);                                                                                                     // 466
});                                                                                                       // 467
                                                                                                          // 468
Tinytest.add('observe-sequence - number arrays', function (test) {                                        // 469
  var seq = [1, 1, 2];                                                                                    // 470
  var dep = new Deps.Dependency;                                                                          // 471
                                                                                                          // 472
  runOneObserveSequenceTestCase(test, function () {                                                       // 473
    dep.depend();                                                                                         // 474
    return seq;                                                                                           // 475
  }, function () {                                                                                        // 476
    seq = [1, 3, 2, 3];                                                                                   // 477
    dep.changed();                                                                                        // 478
  }, [                                                                                                    // 479
    {addedAt: [1, 1, 0, null]},                                                                           // 480
    {addedAt: [{NOT: 1}, 1, 1, null]},                                                                    // 481
    {addedAt: [2, 2, 2, null]},                                                                           // 482
    {removedAt: [{NOT: 1}, 1, 1]},                                                                        // 483
    {addedAt: [3, 3, 1, 2]},                                                                              // 484
    {addedAt: [{NOT: 3}, 3, 3, null]}                                                                     // 485
  ]);                                                                                                     // 486
});                                                                                                       // 487
                                                                                                          // 488
Tinytest.add('observe-sequence - cursor to other cursor, same collection', function (test) {              // 489
  var dep = new Deps.Dependency;                                                                          // 490
  var coll = new Meteor.Collection(null);                                                                 // 491
  coll.insert({_id: "13", foo: 1});                                                                       // 492
  coll.insert({_id: "37", foo: 2});                                                                       // 493
  var cursor = coll.find({foo: 1});                                                                       // 494
  var seq = cursor;                                                                                       // 495
                                                                                                          // 496
  runOneObserveSequenceTestCase(test, function () {                                                       // 497
    dep.depend();                                                                                         // 498
    return seq;                                                                                           // 499
  }, function () {                                                                                        // 500
    var newCursor = coll.find({foo: 2});                                                                  // 501
    seq = newCursor;                                                                                      // 502
    dep.changed();                                                                                        // 503
    Deps.flush();                                                                                         // 504
    coll.insert({_id: "38", foo: 1});                                                                     // 505
    coll.insert({_id: "39", foo: 2});                                                                     // 506
  }, [                                                                                                    // 507
    {addedAt: ["13", {_id: "13", foo: 1}, 0, null]},                                                      // 508
    {removedAt: ["13", {_id: "13", foo: 1}, 0]},                                                          // 509
    {addedAt: ["37", {_id: "37", foo: 2}, 0, null]},                                                      // 510
    {addedAt: ["39", {_id: "39", foo: 2}, 1, null]}                                                       // 511
  ]);                                                                                                     // 512
});                                                                                                       // 513
                                                                                                          // 514
////////////////////////////////////////////////////////////////////////////////////////////////////////////

}).call(this);
